using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Reflection;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Tcp;
using System.Runtime.Remoting.Lifetime;
using System.Runtime.Serialization;
using System.Threading;
using Server.Accounting;
using Server.Engines.Lua;
using LuaInterface;
using RemoteScripting;

namespace Server.Engines.Lua
{
	[Serializable]
	public class LuaCode : MarshalByRefObject, ICode
	{
		private string m_Code;
		private int m_Tag;

		private List<ILuaCodeConsumer> m_Consumers = new List<ILuaCodeConsumer>();

		public string Code
		{
			get { return m_Code; }
			set
			{
				m_Code = value;

				foreach ( ILuaCodeConsumer consumer in m_Consumers )
				{
					consumer.OnScriptChanged();
				}
			}
		}

		public int Tag
		{
			get { return m_Tag; }
			set { m_Tag = value; }
		}

		public LuaCode()
		{
		}

		public override object InitializeLifetimeService()
		{
			return null;
		}

		public string Test( string code )
		{
			LuaInterface.Lua lua = new LuaInterface.Lua();

			string result = "Executed successfully!";

			try
			{
				lua.DoString( code );
			}
			catch ( Exception e )
			{
				result = e.Message;
			}

			return result;
		}

		public void Subscribe( ILuaCodeConsumer consumer )
		{
			if ( consumer != null && !m_Consumers.Contains( consumer ) )
			{
				m_Consumers.Add( consumer );

				consumer.OnScriptChanged();
			}
		}

		public void Unsubscribe( ILuaCodeConsumer consumer )
		{
			if ( consumer != null && m_Consumers.Contains( consumer ) )
			{
				m_Consumers.Remove( consumer );

				consumer.OnLostSubscription();
			}
		}

		public void Dispose()
		{
			foreach ( ILuaCodeConsumer consumer in m_Consumers )
			{
				consumer.OnLostSubscription();
			}
		}

		public void Serialize( GenericWriter writer )
		{
			writer.Write( (int)1 );

			writer.Write( (string)m_Code );
		}

		public void Deserialize( GenericReader reader )
		{
			int version = reader.ReadInt();

			switch ( version )
			{
				case 1:
					{
						m_Code = reader.ReadString();

						break;
					}
			}
		}
	}

	[Serializable]
	public class LuaCodeList : MarshalByRefObject, ICodeList
	{
		private Dictionary<string, LuaCode> m_Items = new Dictionary<string, LuaCode>();

		public int Count
		{
			get { return m_Items.Count; }
		}

		public List<string> Names
		{
			get
			{
				return new List<string>( m_Items.Keys );
			}
		}

		public ICode this[string name]
		{
			get
			{
				if ( m_Items.ContainsKey( name ) )
				{
					return m_Items[name];
				}

				return null;
			}
		}

		public LuaCodeList()
		{
		}

		public override object InitializeLifetimeService()
		{
			return null;
		}

		public ICode Add( string name )
		{
			if ( !m_Items.ContainsKey( name ) )
			{
				LuaCode item = new LuaCode();

				m_Items.Add( name, item );

				return item;
			}

			return null;
		}

		public bool Delete( string name )
		{
			if ( m_Items.ContainsKey( name ) )
			{
				ICode item = m_Items[name];

				m_Items.Remove( name );

				return true;
			}

			return false;
		}

		public void Serialize( GenericWriter writer )
		{
			writer.Write( (int)1 );

			writer.Write( (int)m_Items.Count );

			lock ( m_Items )
			{
				foreach ( string name in m_Items.Keys )
				{
					writer.Write( (string)name );

					LuaCode item = m_Items[name];

					item.Serialize( writer );
				}
			}
		}

		public void Deserialize( GenericReader reader )
		{
			int version = reader.ReadInt();

			switch ( version )
			{
				case 1:
					{
						int count = reader.ReadInt();

						for ( int i = 0; i < count; i++ )
						{
							string name = reader.ReadString();
							LuaCode item = new LuaCode();

							item.Deserialize( reader );
							m_Items.Add( name, item );
						}

						break;
					}
			}
		}
	}

	[Serializable]
	public class LuaCodeBase : MarshalByRefObject, ICodeBase
	{
		private static int m_Port = 8093;

		private static Dictionary<Guid, IAccount> m_Sessions = new Dictionary<Guid, IAccount>();
		private static LuaCodeList m_Scripts = new LuaCodeList();		

		public static LuaCodeList Scripts
		{
			get { return m_Scripts; }
			set { m_Scripts = value; }
		}

		private static bool LoadCodeList( LuaCodeList list, string filename )
		{
			try
			{
				if ( File.Exists( filename ) )
				{
					using ( FileStream bin = new FileStream( filename, FileMode.Open, FileAccess.Read, FileShare.Read ) )
					{
						GenericReader reader = new BinaryFileReader( new BinaryReader( bin ) );

						list.Deserialize( reader );

						return true;
					}
				}
			}
			catch
			{
			}

			return false;
		}

		private static bool SaveCodeList( LuaCodeList list, string filename )
		{
			try
			{
				GenericWriter writer = new BinaryFileWriter( filename, true );

				m_Scripts.Serialize( writer );

				writer.Close();

				return true;
			}
			catch
			{
				return false;
			}
		}

		public static void Configure()
		{
			EventSink.WorldSave += new WorldSaveEventHandler( OnWorldSave );

			Console.Write( "Lua CodeBase: Loading..." );

			LoadCodeList( m_Scripts, Path.Combine( "Saves/Lua", "Scripts.bin" ) );

			Console.WriteLine( "done ({0} scripts)", m_Scripts.Count );
		}

		public static void Initialize()
		{	
			TcpServerChannel channel = new TcpServerChannel( "Lua CodeBase", m_Port, new BinaryServerFormatterSinkProvider() );

			ChannelServices.RegisterChannel( channel, false );
			
			RemotingConfiguration.RegisterWellKnownServiceType(
				typeof( LuaCodeBase ),
				"LuaCodeBase",
				WellKnownObjectMode.Singleton );

			Console.WriteLine( "Lua CodeBase: Listening on port {0}", m_Port );
		}

		public static void OnWorldSave( WorldSaveEventArgs e )
		{
			if ( !Directory.Exists( "Saves/Lua" ) )
			{
				Directory.CreateDirectory( "Saves/Lua" );
			}

			SaveCodeList( m_Scripts, Path.Combine( "Saves/Lua", "Scripts.bin" ) );
		}		

		public LuaCodeBase()
		{
		}

		public override object InitializeLifetimeService()
		{
			return null;
		}

		public Guid SignIn( string username, string password )
		{
			IAccount account = Accounts.GetAccount( username );

			if ( account != null && account.CheckPassword( password ) )
			{
				Console.WriteLine( "Lua CodeBase: Client signed in '{0}'", username );

				if ( account.AccessLevel >= AccessLevel.GameMaster )
				{
					Guid session = Guid.NewGuid();

					m_Sessions.Add( session, account as IAccount );

					return session;
				}
				else
				{
					Console.WriteLine( "Lua CodeBase: Access level is low '{0}'", username );
				}
			}
			else
			{
				Console.WriteLine( "Lua CodeBase: Invalid cridentials for '{0}'", username );
			}

			return Guid.Empty;
		}

		public bool IsLoggedIn( Guid session )
		{
			if ( session == Guid.Empty )
				return false;

			return m_Sessions.ContainsKey( session );
		}

		public ICodeList GetScripts( Guid session )
		{
			if ( !IsLoggedIn( session ) )
				return null;

			return m_Scripts;
		}

		public void SignOut( Guid session )
		{
			if ( !IsLoggedIn( session ) )
				return;

			Console.WriteLine( "Lua CodeBase: Client signed out '{0}'", m_Sessions[session] );

			m_Sessions.Remove( session );
		}
	}
}
